<?php

namespace App\Http\Controllers\Manager\NewsandEvents;

use App\Http\Controllers\Controller;
use App\Models\NewsAndEvents;
use App\Models\NewsAndEventsCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Spatie\Activitylog\Models\Activity;
use Illuminate\Http\File as HttpFile;

class NewsandEventController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:manager_news-event_list', ['only' => ['index', 'getIndex']]);
        $this->middleware('permission:manager_news-event_activity-log', ['only' => ['getActivity', 'getActivityLog']]);
        $this->middleware('permission:manager_news-event_activity-log-trash', ['only' => ['getTrashActivity', 'getTrashActivityLog']]);
        $this->middleware('permission:manager_news-event_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:manager_news-event_show', ['only' => ['show']]);
        $this->middleware('permission:manager_news-event_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:manager_news-event_delete', ['only' => ['destroy']]);
        $this->middleware('permission:manager_news-event_status', ['only' => ['sessionStatus']]);
    }

    public function index()
    {
        $data = [
            'page_title' => 'News and Events',
            'p_title' => 'News and Events',
            'p_summary' => 'List of News and Events',
            'p_description' => null,
            'url' => route('manager.news-and-event.create'),
            'url_text' => 'Add New',
            'trash' => route('manager.get.news-and-event-activity-trash'),
            'trash_text' => 'View Trash',
        ];
        return view('manager.news-and-events.index')->with($data);
    }

    /**
     * Display a listing of the resource.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function getIndex(Request $request)
    {
        ## Read value
        $draw = $request->get('draw');
        $start = $request->get("start");
        $rowperpage = $request->get("length"); // Rows display per page

        $columnIndex_arr = $request->get('order');
        $columnName_arr = $request->get('columns');
        $order_arr = $request->get('order');
        $search_arr = $request->get('search');

        $columnIndex = $columnIndex_arr[0]['column']; // Column index
        $columnName = $columnName_arr[$columnIndex]['data']; // Column name
        $columnSortOrder = $order_arr[0]['dir']; // asc or desc
        $searchValue = $search_arr['value']; // Search value

        // Total records
        $totalRecords = NewsAndEvents::select('news_and_events.*')->count();
        // Total records with filter
        $totalRecordswithFilter = NewsAndEvents::select('news_and_events.*')
            ->where(function ($q) use ($searchValue) {
                $q->where('news_and_events.title', 'like', '%' . $searchValue . '%');
            })
            ->count();
        // Fetch records
        $records = NewsAndEvents::select('news_and_events.*')
            ->where(function ($q) use ($searchValue) {
                $q->where('news_and_events.title', 'like', '%' . $searchValue . '%');
            })
            ->skip($start)
            ->take($rowperpage)
            ->orderBy($columnName, $columnSortOrder)
            ->get();

        $data_arr = array();

        foreach ($records as $record) {
            $id = $record->id;
            $title = $record->title;
            $date = $record->date;
            $type = $record->type;
            $status = $record->status;

            $data_arr[] = array(
                "id" => $id,
                "title" => $title,
                "date" => $date,
                "type" => $type,
                "status" => $status,
            );
        }
        $response = array(
            "draw" => intval($draw),
            "iTotalRecords" => $totalRecords,
            "iTotalDisplayRecords" => $totalRecordswithFilter,
            "aaData" => $data_arr
        );
        echo json_encode($response);
        exit;
    }

    /**
     * Display a listing of the resource.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function getIndexSelect(Request $request)
    {
        $data = [];

        if ($request->has('q')) {
            $search = $request->q;
            $data = NewsAndEvents::select('news_and_events.id as id', 'news_and_events.title as title')
                ->where('news_and_events.status', '1')
                ->where(function ($q) use ($search) {
                    where("news_and_events.status = 1");
                    $q->where('news_and_events.title', 'like', '%' . $search . '%');
                })->get();
        }

        return response()->json($data);

    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $data = array(
            'page_title' => 'News and Events',
            'p_title' => 'News and Events',
            'p_summary' => 'Add News and Events',
            'p_description' => null,
            'method' => 'POST',
            'action' => route('manager.news-and-event.store'),
            'url' => route('manager.news-and-event.index'),
            'url_text' => 'View All',
            'enctype' => 'multipart/form-data' // (Default)Without attachment
//            'enctype' => 'application/x-www-form-urlencoded', // With attachment like file or images in form
        );
        return view('manager.news-and-events.create')->with($data);
    }

    /**
     * Store a newly created resource in storage.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        $this->validate($request, [
            'title' => 'required|',
            'date' => 'required|',
            'description' => 'required|',
            'news_and_events_category_id' => 'required|',
            'status' => 'required|',
            'type' => 'required|',
            'summary' => 'required|',
            'thumbnail' => 'required|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'cover_image' => 'required|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        //

        $thumbnail_file_arr = $request->file('thumbnail');
        $thumbnailExtension = '';
        $thumbnailSize = '';

        if (!empty($request->hasFile('thumbnail'))) {
//            foreach ($thumbnail_file_arr as $uploaded_file) {

            //Uploading Document Attachment
            if ($request->hasFile('thumbnail')) {
                $imageOriginalName = $thumbnail_file_arr->getClientOriginalName();
                $thumbnailExtension = $thumbnail_file_arr->getClientOriginalExtension();
                $thumbnailSize = $thumbnail_file_arr->getSize();
                $filename = date('Y') . '/' . date('m') . '/' . Str::slug($request->title) . '-' . date('d') . time() . '-' . rand(0, 999999);
                $filename = $filename . '.' . $thumbnailExtension;
                $folderPath = 'News-and-Events/thumbnail';
                Storage::disk('public')->putFileAs($folderPath, new HttpFile($thumbnail_file_arr), $filename);
                $thumbnail = $filename;
            }
        } else {
            $thumbnail = null;
        }

        $cover_file_arr = $request->file('cover_image');
        $coverExtension = '';
        $coverSize = '';

        if (!empty($request->hasFile('cover_image'))) {
//            foreach ($cover_file_arr as $uploaded_file) {

            //Uploading Document Attachment
            if ($request->hasFile('cover_image')) {
                $imageOriginalName = $cover_file_arr->getClientOriginalName();
                $coverExtension = $cover_file_arr->getClientOriginalExtension();
                $coverSize = $cover_file_arr->getSize();
                $filename = date('Y') . '/' . date('m') . '/' . Str::slug($request->title) . '-' . date('d') . time() . '-' . rand(0, 999999);
                $filename = $filename . '.' . $coverExtension;
                $folderPath = 'News-and-Events/Cover';
                Storage::disk('public')->putFileAs($folderPath, new HttpFile($cover_file_arr), $filename);
                $cover = $filename;
            }
        } else {
            $cover = null;
        }
        $arr = [
            'title' => $request->input('title'),
            'slug' => $this->createSlug($request->title),
            'date' => $request->input('date'),
            'description' => $request->input('description'),
            'news_and_events_category_id' => $request->input('news_and_events_category_id'),
            'status' => $request->input('status'),
            'type' => $request->input('type'),
            'summary' => $request->input('summary'),
            'thumbnail' => $thumbnail,
            'cover_image' => $cover,
        ];

        $record = NewsAndEvents::create($arr);
        $messages = [
            array(
                'message' => 'Record created successfully',
                'message_type' => 'success'
            ),
        ];
        Session::flash('messages', $messages);

        return redirect()->route('manager.news-and-event.index');
    }

    public function createSlug($title, $id = 0)
    {
        $slug = Str::slug($title);
        $originalSlug = $slug;
        $count = 1;

        // Check if a record with the slug exists in the database
        while (NewsAndEvents::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $count++;
        }

        return $slug;

    }

    /**
     * Display the specified resource.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function show(string $id)
    {
        $record = NewsAndEvents::select('news_and_events.*')
            ->where('id', '=', $id)
            ->first();

        if (empty($record)) {
            abort(404, 'NOT FOUND');
        }
        $category = NewsAndEventsCategory::get();

        // Add activity logs
        $user = Auth::user();
        activity('News and Events')
            ->performedOn($record)
            ->causedBy($user)
            ->event('viewed')
            ->withProperties(['attributes' => ['name' => $record->name]])
            ->log('viewed');
        //Data Array
        $data = array(
            'page_title' => 'News and Events',
            'p_title' => 'News and Events',
            'p_summary' => 'Show News and Events',
            'p_description' => null,
            'method' => 'POST',
            'action' => route('manager.news-and-event.update', $record->id),
            'url' => route('manager.news-and-event.index'),
            'url_text' => 'View All',
            'data' => $record,
            'categories' => $category,
            'enctype' => 'multipart/form-data' // (Default)Without attachment
//            'enctype' => 'application/x-www-form-urlencoded', // With attachment like file or images in form
        );
        return view('manager.news-and-events.show')->with($data);
    }

    /**
     * Display the specified resource Activity.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function getActivity(string $id)
    {
        //Data Array
        $data = array(
            'page_title' => 'News and Events Activity',
            'p_title' => 'News and Events Activity',
            'p_summary' => 'Show News and Events Activity',
            'p_description' => null,
            'url' => route('manager.news-and-event.index'),
            'url_text' => 'View All',
            'id' => $id,
        );
        return view('manager.news-and-events.activity')->with($data);
    }

    /**
     * Display the specified resource Activity Logs.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function getActivityLog(Request $request, string $id)
    {
        ## Read value
        $draw = $request->get('draw');
        $start = $request->get("start");
        $rowperpage = $request->get("length"); // Rows display per page

        $columnIndex_arr = $request->get('order');
        $columnName_arr = $request->get('columns');
        $order_arr = $request->get('order');
        $search_arr = $request->get('search');

        $columnIndex = $columnIndex_arr[0]['column']; // Column index
        $columnName = $columnName_arr[$columnIndex]['data']; // Column name
        $columnSortOrder = $order_arr[0]['dir']; // asc or desc
        $searchValue = $search_arr['value']; // Search value

        // Total records
        $totalRecords = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where('activity_log.subject_id', $id)
            ->count();

        // Total records with filter
        $totalRecordswithFilter = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_id', $id)
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where(function ($q) use ($searchValue) {
                $q->where('activity_log.description', 'like', '%' . $searchValue . '%')
                    ->orWhere('users.name', 'like', '%' . $searchValue . '%');
            })
            ->count();

        // Fetch records
        $records = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_id', $id)
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where(function ($q) use ($searchValue) {
                $q->where('activity_log.description', 'like', '%' . $searchValue . '%')
                    ->orWhere('users.name', 'like', '%' . $searchValue . '%');
            })
            ->skip($start)
            ->take($rowperpage)
            ->orderBy($columnName, $columnSortOrder)
            ->get();


        $data_arr = array();

        foreach ($records as $record) {
            $id = $record->id;
            $attributes = (!empty($record->properties['attributes']) ? $record->properties['attributes'] : '');
            $old = (!empty($record->properties['old']) ? $record->properties['old'] : '');
            $current = '<ul class="list-unstyled">';
            //Current
            if (!empty($attributes)) {
                foreach ($attributes as $key => $value) {
                    if (is_array($value)) {
                        $current .= '<li>';
                        $current .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $current .= '</li>';
                    } else {
                        $current .= '<li>';
                        $current .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $current .= '</li>';
                    }
                }
            }
            $current .= '</ul>';
            //Old
            $oldValue = '<ul class="list-unstyled">';
            if (!empty($old)) {
                foreach ($old as $key => $value) {
                    if (is_array($value)) {
                        $oldValue .= '<li>';
                        $oldValue .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $oldValue .= '</li>';
                    } else {
                        $oldValue .= '<li>';
                        $oldValue .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $oldValue .= '</li>';
                    }
                }
            }
            //updated at
            $updated = 'Updated:' . $record->updated_at->diffForHumans() . '<br> At:' . $record->updated_at->isoFormat('llll');
            $oldValue .= '</ul>';
            //Causer
            $causer = isset($record->causer) ? $record->causer : '';
            $type = $record->description;
            $data_arr[] = array(
                "id" => $id,
                "current" => $current,
                "old" => $oldValue,
                "updated" => $updated,
                "causer" => $causer,
                "type" => $type,
            );
        }
        $response = array(
            "draw" => intval($draw),
            "iTotalRecords" => $totalRecords,
            "iTotalDisplayRecords" => $totalRecordswithFilter,
            "aaData" => $data_arr
        );

        echo json_encode($response);
        exit;
    }

    /**
     * Display the trash resource Activity.
     * @return \Illuminate\Http\Response
     */
    public function getTrashActivity()
    {
        //Data Array
        $data = array(
            'page_title' => 'News and Events Activity',
            'p_title' => 'News and Events Activity',
            'p_summary' => 'Show News and Events Trashed Activity',
            'p_description' => null,
            'url' => route('manager.news-and-event.index'),
            'url_text' => 'View All',
        );
        return view('manager.news-and-events.trash')->with($data);
    }

    /**
     * Display the trash resource Activity Logs.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function getTrashActivityLog(Request $request)
    {
        ## Read value
        $draw = $request->get('draw');
        $start = $request->get("start");
        $rowperpage = $request->get("length"); // Rows display per page

        $columnIndex_arr = $request->get('order');
        $columnName_arr = $request->get('columns');
        $order_arr = $request->get('order');
        $search_arr = $request->get('search');

        $columnIndex = $columnIndex_arr[0]['column']; // Column index
        $columnName = $columnName_arr[$columnIndex]['data']; // Column name
        $columnSortOrder = $order_arr[0]['dir']; // asc or desc
        $searchValue = $search_arr['value']; // Search value

        // Total records
        $totalRecords = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where('activity_log.event', 'deleted')
            ->count();

        // Total records with filter
        $totalRecordswithFilter = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where('activity_log.event', 'deleted')
            ->where(function ($q) use ($searchValue) {
                $q->where('activity_log.description', 'like', '%' . $searchValue . '%')
                    ->orWhere('users.name', 'like', '%' . $searchValue . '%');
            })
            ->count();

        // Fetch records
        $records = Activity::select('activity_log.*', 'users.name as causer')
            ->leftJoin('users', 'users.id', 'activity_log.causer_id')
            ->leftJoin('news_and_events', 'news_and_events.id', 'activity_log.subject_id')
            ->where('activity_log.subject_type', NewsAndEvents::class)
            ->where('activity_log.event', 'deleted')
            ->where(function ($q) use ($searchValue) {
                $q->where('activity_log.description', 'like', '%' . $searchValue . '%')
                    ->orWhere('users.name', 'like', '%' . $searchValue . '%');
            })
            ->skip($start)
            ->take($rowperpage)
            ->orderBy($columnName, $columnSortOrder)
            ->get();


        $data_arr = array();

        foreach ($records as $record) {
            $id = $record->id;
            $attributes = (!empty($record->properties['attributes']) ? $record->properties['attributes'] : '');
            $old = (!empty($record->properties['old']) ? $record->properties['old'] : '');
            $current = '<ul class="list-unstyled">';
            //Current
            if (!empty($attributes)) {
                foreach ($attributes as $key => $value) {
                    if (is_array($value)) {
                        $current .= '<li>';
                        $current .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $current .= '</li>';
                    } else {
                        $current .= '<li>';
                        $current .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $current .= '</li>';
                    }
                }
            }
            $current .= '</ul>';
            //Old
            $oldValue = '<ul class="list-unstyled">';
            if (!empty($old)) {
                foreach ($old as $key => $value) {
                    if (is_array($value)) {
                        $oldValue .= '<li>';
                        $oldValue .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $oldValue .= '</li>';
                    } else {
                        $oldValue .= '<li>';
                        $oldValue .= '<i class="fas fa-angle-right"></i> <em></em>' . $key . ': <mark>' . $value . '</mark>';
                        $oldValue .= '</li>';
                    }
                }
            }
            //updated at
            $updated = 'Updated:' . $record->updated_at->diffForHumans() . '<br> At:' . $record->updated_at->isoFormat('llll');
            $oldValue .= '</ul>';
            //Causer
            $causer = isset($record->causer) ? $record->causer : '';
            $type = $record->description;
            $data_arr[] = array(
                "id" => $id,
                "current" => $current,
                "old" => $oldValue,
                "updated" => $updated,
                "causer" => $causer,
                "type" => $type,
            );
        }
        $response = array(
            "draw" => intval($draw),
            "iTotalRecords" => $totalRecords,
            "iTotalDisplayRecords" => $totalRecordswithFilter,
            "aaData" => $data_arr
        );

        echo json_encode($response);
        exit;
    }

    /**
     * Show the form for editing the specified resource.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function edit(string $id)
    {
        $record = NewsAndEvents::select('news_and_events.*')
            ->where('id', '=', $id)
            ->first();
//        dd($record);

        if (empty($record)) {
            abort(404, 'NOT FOUND');
        }
        $category = NewsAndEventsCategory::get();
        $data = array(
            'page_title' => 'News and Events',
            'p_title' => 'News and Events',
            'p_summary' => 'Edit News and Events',
            'p_description' => null,
            'method' => 'POST',
            'action' => route('manager.news-and-event.update', $record->id),
            'url' => route('manager.news-and-event.index'),
            'url_text' => 'View All',
            'data' => $record,
            'categories' => $category,
            'enctype' => 'multipart/form-data' // (Default)Without attachment
//            'enctype' => 'application/x-www-form-urlencoded', // With attachment like file or images in form
        );
        return view('manager.news-and-events.edit')->with($data);
    }

    /**
     * Update the specified resource in storage.
     * @param String_ $id
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, string $id)
    {
        $record = NewsAndEvents::select('news_and_events.*')
            ->where('id', '=', $id)
            ->first();
        if (empty($record)) {
            abort(404, 'NOT FOUND');
        }
        $this->validate($request, [
            'title' => 'required',
            'date' => 'required',
            'description' => 'required',
            'type' => 'required',
            'news_and_events_category_id' => 'required',
            'summary' => 'required',
            'thumbnail' => 'required|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'cover_image' => 'required|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        //


        if (!empty($request->hasFile('thumbnail'))) {

            //            foreach ($thumbnail_file_arr as $uploaded_file) {
            //Unlink previous image
            if (isset($record) && $record->thumbnail) {
//                dd('123');
                $prevImage = Storage::disk('public')->path('News-and-Events/thumbnail/' . $record->thumbnail);
                if (File::exists($prevImage)) { // unlink or remove previous image from folder
                    File::delete($prevImage);
                }
            }
            $thumbnail_file_arr = $request->file('thumbnail');
            $thumbnailExtension = '';
            $thumbnailSize = '';
            //Uploading Document Attachment
            if ($request->hasFile('thumbnail')) {
                $imageOriginalName = $thumbnail_file_arr->getClientOriginalName();
                $thumbnailExtension = $thumbnail_file_arr->getClientOriginalExtension();
                $thumbnailSize = $thumbnail_file_arr->getSize();
                $filename = date('Y') . '/' . date('m') . '/' . Str::slug($request->title) . '-' . date('d') . time() . '-' . rand(0, 999999);
                $filename = $filename . '.' . $thumbnailExtension;
                $folderPath = 'News-and-Events/thumbnail';
                Storage::disk('public')->putFileAs($folderPath, new HttpFile($thumbnail_file_arr), $filename);
                $thumbnail = $filename;
            }
        } else {
            $thumbnail = $record->thumbnail;

        }


        if (!empty($request->hasFile('cover_image'))) {
//            foreach ($cover_file_arr as $uploaded_file) {
            if (isset($record) && $record->cover_image) {
                $prevImage = Storage::disk('public')->path('News-and-Events/cover/' . $record->cover_image);
                if (File::exists($prevImage)) { // unlink or remove previous image from folder
                    File::delete($prevImage);
                }
            }
            $cover_file_arr = $request->file('cover_image');
            $coverExtension = '';
            $coverSize = '';
            //Uploading Document Attachment
            if ($request->hasFile('cover_image')) {
                $imageOriginalName = $cover_file_arr->getClientOriginalName();
                $coverExtension = $cover_file_arr->getClientOriginalExtension();
                $coverSize = $cover_file_arr->getSize();
                $filename = date('Y') . '/' . date('m') . '/' . Str::slug($request->title) . '-' . date('d') . time() . '-' . rand(0, 999999);
                $filename = $filename . '.' . $coverExtension;
                $folderPath = 'News-and-Events/Cover';
                Storage::disk('public')->putFileAs($folderPath, new HttpFile($cover_file_arr), $filename);
                $cover = $filename;
            }

        } else {
            $cover = $record->cover_image;
        }
        $arr = [
            'title' => $request->input('title'),
            'date' => $request->input('date'),
            'description' => $request->input('description'),
            'news_and_events_category_id' => $request->input('news_and_events_category_id'),
            'summary' => $request->input('summary'),
            'type' => $request->input('type'),
            'thumbnail' => $thumbnail,
            'cover_image' => $cover,
        ];
        $record->update($arr);
        $messages = [
            array(
                'message' => 'Record updated successfully',
                'message_type' => 'success'
            ),
        ];
        Session::flash('messages', $messages);

        return redirect()->route('manager.news-and-event.index');
    }

    /**
     * Remove the specified resource from storage.
     * @param String_ $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(string $id)
    {
        $record = NewsAndEvents::select('news_and_events.*')
            ->where('id', '=', $id)
            ->first();
        if (empty($record)) {
            abort(404, 'NOT FOUND');
        }
        $image_path = Storage::disk('public')->path('News-and-Events/thumbnail/' . $record->thumbnail);
        if (File::exists($image_path)) {
            File::delete($image_path);
        }
        $thumbnail = Storage::disk('public')->path('News-and-Events/Cover/' . $record->cover_image);
        if (File::exists($thumbnail)) {
            File::delete($thumbnail);
        }

        $record->delete();

        $messages = [
            array(
                'message' => 'Record deleted successfully',
                'message_type' => 'success'
            ),
        ];
        Session::flash('messages', $messages);

        return redirect()->route('manager.news-and-event.index');
    }

    public function sessionStatus(string $id)

    {
        $record = NewsAndEvents::find($id);


        if ($record->status == '1') {
            $value = '0';
        } else {
            $value = '1';
        }
        $arr = [
            'status' => $value,
        ];

        $record->update($arr);
        $messages = [
            array(
                'message' => 'Record updated successfully',
                'message_type' => 'success'
            ),
        ];
        $user = Auth::user();
        activity('News and Events')
            ->performedOn($record)
            ->causedBy($user)
            ->event('status')
            ->withProperties(['attributes' => ['status' => $record->status]])
            ->log('Status Changed');
        Session::flash('messages', $messages);
        return redirect()->route('manager.news-and-event.index');

    }
}
